<?php

if (!defined('ABSPATH')) {
  exit; // Exit if accessed directly.
}

class Ultimate_AI_User_Controller
{
  private $user_manager;
  private $theme_settings_manager;

  public function __construct()
  {
    $this->user_manager = new Ultimate_AI_User_Manager();
    $this->theme_settings_manager = new Ultimate_AI_Theme_Setting_Manager();

    add_action('wp_ajax_ultimate_ai_update_user_role', [$this, 'ultimate_ai_update_user_role']);
    add_action('wp_ajax_ultimate_ai_add_user', [$this, 'ultimate_ai_add_user']);
    add_action('wp_ajax_ultimate_ai_delete_user', [$this, 'ultimate_ai_delete_user']);
    add_action('wp_ajax_ultimate_ai_get_current_user', [$this, 'ultimate_ai_get_current_user']);
    add_action('wp_ajax_ultimate_ai_filter_users', [$this, 'ultimate_ai_filter_users']);
    add_action('wp_ajax_ultimate_ai_get_subscribers', [$this, 'ultimate_ai_get_subscribers']);
    add_action('wp_ajax_ultimate_ai_add_subscriber', [$this, 'ultimate_ai_add_subscriber']);
    add_action('wp_ajax_ultimate_ai_logout_user', [$this, 'ultimate_ai_logout_user']);
    add_action('wp_ajax_reset_pass', [$this, 'ultimate_ai_change_pass']);
    add_action('wp_ajax_nopriv_reset_pass', [$this, 'ultimate_ai_change_pass']);
    add_action('wp_ajax_send_pass_reset', [$this, 'ultimate_ai_send_pass_reset']);
    add_action('wp_ajax_nopriv_send_pass_reset', [$this, 'ultimate_ai_send_pass_reset']);
    add_action('wp_ajax_get_user_settings', [$this, 'ultimate_ai_get_user_settings']);
    add_action('wp_ajax_nopriv_get_user_settings', [$this, 'ultimate_ai_get_user_settings']);
    add_action('wp_ajax_ultimate_ai_update_user_settings', [$this, 'ultimate_ai_update_user_settings']);
    add_action('wp_ajax_ultimate_ai_update_user_profile_photo', [$this, 'ultimate_ai_update_user_profile_photo']);
    add_action('wp_ajax_nopriv_ultimate_ai_register_user', [$this, 'ultimate_ai_register_user']);
    add_action('wp_ajax_nopriv_ultimate_ai_login_user', [$this, 'ultimate_ai_login_user']);
    add_action('wp_ajax_nopriv_ultimate_ai_user_exist_by', [$this, 'ultimate_ai_user_exist_by']);
    add_action('wp_ajax_nopriv_ultimate_ai_register_or_login_with_google', [$this, 'ultimate_ai_register_or_login_with_google']);
    add_action('wp_ajax_ultimate_ai_user_exist_by', [$this, 'ultimate_ai_user_exist_by']);
    add_action('wp_ajax_ultimate_ai_change_password', [$this, 'ultimate_ai_change_password']);
    add_action('wp_ajax_ultimate_ai_verify_user_email', [$this, 'ultimate_ai_verify_user_email']);
    add_action('wp_ajax_ultimate_user_integration', [$this, 'ultimate_user_integration']);
    add_action('wp_ajax_ultimate_ai_access_integration', [$this, 'ultimate_ai_access_integration']);
    add_action('wp_ajax_ultimate_ai_update_tokens', [$this, 'ultimate_ai_update_tokens']);
    add_action('wp_ajax_ultimate_ai_update_email_verification_status', [$this, 'ultimate_ai_update_email_verification_status']);

    //user meta update
    add_action('wp_ajax_ultimate_ai_update_user_meta', [$this, 'ultimate_ai_update_user_meta']);

    // for login/logout redirection
    add_action('template_redirect', [$this, 'ultimate_ai_redirect_user_to_login_page']);
  }

  public function ultimate_ai_update_user_meta()
  {
    if (
      !isset($_POST["keys"]) ||
      empty($_POST["keys"]) ||
      empty($_POST["values"]) ||
      empty($_POST["values"])
    ) {
      wp_send_json(["data" => "Invalid data"], 200);
    }

    $keys = map_deep($_POST["keys"], "sanitize_key");
    $values = map_deep($_POST["values"], "sanitize_text_field");
    $user = wp_get_current_user();

    $new_values = [];
    for ($i = 0; $i < count($keys); $i++) {
      $new_values[$keys[$i]] = $values[$i];
    }

    $is_success = $this->user_manager->update_user_meta($user, $new_values);
    wp_send_json(["status" => $is_success]);
  }

  public function ultimate_ai_update_email_verification_status()
  {
    $this->restrict_req_for_admin();

    $user_id = $_POST['user_id'];
    $user = get_user_by('ID', $user_id);

    if (!isset($user_id) || empty($user_id) || $user_id == 0) {
      wp_send_json([
        'status' => 'error',
        'message' => esc_html__('User id not found', ULTIMATE_AI_SLUG)
      ], 200);
    }
    $updated = update_user_meta($user_id, 'uai_is_email_verified', 'true');
    if ($updated) {
      wp_send_json([
        'status' => 'success',
        'message' => esc_html__('Email verification success.', ULTIMATE_AI_SLUG)
      ], 200);
    } else {
      wp_send_json([
        'status' => 'error',
        'message' => esc_html__('Email verification failed.', ULTIMATE_AI_SLUG)
      ], 200);
    }
  }

  public function ultimate_ai_update_tokens()
  {
    $this->restrict_req_for_admin();

    $data = $_POST['usage'];

    if ((int) $data['user_id'] == 0) {
      $users = get_users();
      foreach ($users as $user) {
        $user = get_user_by('ID', $user->ID);
        $res = $this->update_token_by_user($user, $data);
      }
      wp_send_json($res, 200);
    }

    if ((int) $data['user_id'] > 0) {
      $user_id = (int) $data['user_id'];

      $user = get_user_by('ID', $user_id);
      $res = $this->update_token_by_user($user, $data);

      wp_send_json($res, 200);
    }
  }

  public function update_token_by_user($user, $data)
  {
    $reset = isset($data['reset']) && $data['reset'] == 'true' ? true : false;

    $full_user = $this->user_manager->get_full_user($user);
    $_limits = $full_user["meta"]["limits"];

    $map_token_types_to_limits_key = ["word" => "words", "image" => "images", "tts" => "tts"];

    foreach ($map_token_types_to_limits_key as $token_type => $limit_key) {
      if (isset($data[$token_type]) && (int) $data[$token_type] > 0 && array_key_exists($limit_key, $_limits)) {

        //modify the limits 
        $_limits[$limit_key] = $reset ? (int) $data[$token_type] : (int) $data[$token_type] + (int) $_limits[$limit_key];

        //update user limits also
        $this->user_manager->update_user_meta($user, ["limits" => $_limits]);

        //update user token amount
        $this->user_manager->update_user_token($user, (int) $data[$token_type], $token_type, 'INCREMENT', $reset);
      }
    }

    return [
      'success' => true
    ];
  }

  public function ultimate_ai_verify_user_email()
  {
    if (!isset($_POST['user_id']) || empty($_POST['user_id'])) {
      wp_send_json([
        'status' => false,
        'message' => esc_html__('User is not logged in', ULTIMATE_AI_SLUG),
      ], 200);
    }
    if (!isset($_POST['verification_key']) || empty($_POST['verification_key'])) {
      wp_send_json([
        'status' => false,
        'message' => esc_html__('Verification key not found.', ULTIMATE_AI_SLUG),
      ], 200);
    }
    if (!is_user_logged_in()) {
      wp_send_json([
        'status' => false,
        'message' => esc_html__('User is not logged in', ULTIMATE_AI_SLUG),
      ], 200);
    }

    $user_id = $_POST['user_id'];
    $verification_key = $_POST['verification_key'];

    $key = get_user_meta($user_id, 'uai_user_verification_key', true);
    if ($key = $verification_key) {
      update_user_meta($user_id, 'uai_is_email_verified', 'true');

      $user = wp_get_current_user();
      if (!$user->ID) {
        wp_send_json(['data' => 'No user found'], 400);
      }
      // remove password form response
      $user->data->user_pass = '';
      $user_manager = new Ultimate_AI_User_Manager();
      $user = $user_manager->get_full_user($user);
      wp_send_json(['data' => $user], 200);
    } else {
      wp_send_json([
        'status' => false,
        'message' => esc_html__('Verification key did not match', ULTIMATE_AI_SLUG),
      ], 200);
    }
  }

  public function ultimate_ai_update_user_role()
  {
    $this->restrict_req_for_admin();
    $this->must_fields_error(['role', 'user_id']);

    $role = sanitize_text_field($_POST['role']);
    $user_id = (int) sanitize_text_field($_POST['user_id']);
    $res = $this->user_manager->update_user_role($role, $user_id);
    if ($res['status'] == true) {
      wp_send_json($res, 200);
    }
    // recach this line then update_user_role failed
    wp_send_json($res, 400);
  }

  public function ultimate_ai_logout_user()
  {
    wp_logout();
    ob_clean();
    wp_send_json(
      [
        'redirect' => site_url(),
        'status' => 200,
      ],
      200
    );
  }

  public function ultimate_ai_delete_user()
  {
    $this->restrict_req_for_admin();
    $user_id = (int) sanitize_text_field($_POST['user_id']);

    $deleted = $this->user_manager->deleteUser($user_id);

    wp_send_json($deleted, 200);
  }

  public function ultimate_ai_add_user()
  {
    $this->restrict_req_for_admin();
    $user = Ultimate_AI_Filter_Controller($_POST['user']);
    $user['user_pass'] = wp_hash_password($user['user_email']);

    $user = $this->user_manager->add_new_user($user);
    wp_send_json($user, 200);
  }

  public function ultimate_ai_filter_users()
  {
    $args = [];
    if (isset($_POST['limit'])) {
      $args['number'] = Ultimate_AI_Filter_Controller($_POST['limit']);
    }
    if (isset($_POST['offset'])) {
      $args['offset'] = Ultimate_AI_Filter_Controller($_POST['offset']);
    }
    if (isset($_POST['fields'])) {
      $args['fields'] = Ultimate_AI_Filter_Controller($_POST['fields']);
    }
    if (isset($_POST['search'])) {
      $args['search'] = '*' . esc_attr($_POST['search']) . '*';
      $args['search_columns'] = [
        'user_login',
        'user_nicename',
        'user_email',
        'user_url',
      ];
    }

    $res = $this->user_manager->filter_users($args);

    wp_send_json($res, 200);
  }

  private function ultimate_ai_give_free_token_upon_registration($user)
  {
    $app_settings = $this->theme_settings_manager->ultimate_ai_get_theme_settings(['uai_app_settings'])['uai_app_settings'];
    $user_data = $this->user_manager->get_full_user($user);
    $limits_meta = $user_data['meta']['limits'] ?? [];

    // image token
    if (!empty($app_settings['free_image_token_on_registration'])) {
      $this->user_manager->update_user_token(
        $user,
        (int) $app_settings['free_image_token_on_registration'],
        'image',
        'INCREMENT',
        true
      );

      // update limits meta
      $limits_meta['images'] = (int) $app_settings['free_image_token_on_registration'];
      $this->user_manager->update_user_meta($user, ['limits' => $limits_meta]);
    }

    // word token
    if (!empty($app_settings['free_word_token_on_registration'])) {
      $this->user_manager->update_user_token(
        $user,
        (int) $app_settings['free_word_token_on_registration'],
        'word',
        'INCREMENT',
        true
      );
      // update limits meta
      $limits_meta['words'] = (int) $app_settings['free_word_token_on_registration'];
      $this->user_manager->update_user_meta($user, ['limits' => $limits_meta]);
    }

    // tts token
    if (!empty($app_settings['free_tts_token_on_registration'])) {
      $this->user_manager->update_user_token(
        $user,
        (int) $app_settings['free_tts_token_on_registration'],
        'tts',
        'INCREMENT',
        true
      );
      // update limits meta
      $limits_meta['tts'] = (int) $app_settings['free_tts_token_on_registration'];
      $this->user_manager->update_user_meta($user, ['limits' => $limits_meta]);
    }
  }

  public function ultimate_ai_access_integration()
  {
    $key = $_POST['key'];
    $access = $this->user_manager->check_user_integration_access($key);
    wp_send_json($access, 200);
  }

  public function ultimate_user_integration()
  {
    if (!function_exists('wp_get_current_user')) {
      include(ABSPATH . "wp-includes/pluggable.php");
    }
    $user = wp_get_current_user();
    $meta_key = $_POST['meta_key'];
    $meta_value = $_POST['meta_value'];
    $is_success = $this->user_manager->update_user_meta($user, [$meta_key => $meta_value]);

    wp_send_json(['data' => ['status' => 'success']]);
  }

  // public function ultimate_user_app_setting()
  // {
  //     $user = wp_get_current_user();
  //     $app_settings = $_POST['data'];

  //     $is_success = $this->user_manager->update_user_meta($user, ['app_settings' => $app_settings]);

  //     if (!$is_success) {
  //         wp_send_json(['data' => 'update information fail!'], 400);
  //     }

  //     wp_send_json(['data' => ['status' => 'success']]);
  // }

  public function ultimate_ai_redirect_user_to_login_page()
  {
    global $wp;
    $app_settings = $this->theme_settings_manager->ultimate_ai_get_theme_settings(['uai_app_settings']);

    $login_page_url = $app_settings['uai_app_settings']['register_page_url'] ?? home_url();
    $login_page_url = $login_page_url ? $login_page_url : home_url();
    $current_url = home_url($wp->request);
    $dashboard_page_id = get_option('ultimate_ai_dashboard_page_id');

    // ONLY REDIRECTION WORK IF ITS A DASHBOARD/SUB-DASHBOARD PAGE OR LOGIN PAGE
    // MAKE SURE LOGIN_PAGE_URL DONT HAVE / AT THE END OF URL WHEN SAVE ON DASHBOARD
    if (is_page($dashboard_page_id) || $login_page_url == $current_url) {
      if (!is_user_logged_in() && $login_page_url != $current_url && !current_user_can('edit_pages')) {
        wp_redirect($login_page_url);
        exit;
      }

      if (is_user_logged_in() && $login_page_url == $current_url && !current_user_can('edit_pages')) {
        wp_redirect(get_the_permalink($dashboard_page_id));
        exit;
      }
    }
  }

  public function ultimate_ai_update_user_settings()
  {
    $first_name = sanitize_text_field($_POST['first_name'] ?? '');
    $last_name = sanitize_text_field($_POST['last_name'] ?? '');
    $display_name = sanitize_text_field($_POST['display_name'] ?? '');
    $user_name = sanitize_text_field($_POST['user_name'] ?? '');
    $country = sanitize_text_field($_POST['country'] ?? '');
    $street_address = sanitize_text_field($_POST['street_address'] ?? '');
    $email = sanitize_email($_POST['email'] ?? '');
    $profile_photo = sanitize_text_field($_POST['profile_photo'] ?? '');
    $phone = sanitize_text_field($_POST['phone'] ?? '');
    $state_province = sanitize_text_field($_POST['state_province'] ?? '');
    $zip_postal_code = sanitize_text_field($_POST['zip_postal_code'] ?? '');
    $user = wp_get_current_user();

    $full_user_info = $this->user_manager->get_full_user($user);
    $profile_photo_added = $full_user_info['meta']['user_info']['profile_photo'];

    $profile_photo = !empty($profile_photo) ? $profile_photo : ($profile_photo_added ?? '');

    $data = [
      'user_info' => [
        'first_name' => $first_name,
        'last_name' => $last_name,
        'display_name' => $display_name,
        'user_name' => $user_name,
        'country' => $country,
        'street_address' => $street_address,
        'email' => $email,
        'profile_photo' => $profile_photo,
        'phone' => $phone,
        'state_province' => $state_province,
        'zip_postal_code' => $zip_postal_code,
      ],
    ];

    $is_success = $this->user_manager->update_user_meta($user, $data);

    if (!$is_success) {
      wp_send_json(['data' => 'update information fail!'], 400);
    }

    wp_send_json(['data' => ['status' => 'success']]);
  }

  public function ultimate_ai_update_user_profile_photo()
  {
    $profile_photo = sanitize_text_field($_POST['profile_photo'] ?? '');
    $social_link = Ultimate_AI_Filter_Controller($_POST['social_link']) ?? '';

    if (is_array($social_link)) {
      foreach ($social_link as $key => $value) {
        $sanitized_value = sanitize_text_field($value);
        $social_link[$key] = $sanitized_value;
      }
    }

    $user = wp_get_current_user();

    $full_user_info = $this->user_manager->get_full_user($user);

    $user_info = $full_user_info['meta']['user_info'];
    $user_info['profile_photo'] = $profile_photo;
    $user_info['social_link'] = $social_link;

    $is_success = $this->user_manager->update_user_meta($user, ['user_info' => $user_info]);

    if (!$is_success) {
      wp_send_json(['data' => 'update information fail!'], 400);
    }
    wp_send_json(['data' => ['status' => 'success']]);
  }

  public function ultimate_ai_get_user_settings()
  {
    if (is_user_logged_in()) {
      $user_id = get_current_user_id();
      $user_meta = get_user_meta($user_id, 'user_settings_panel', true);
      if ($user_meta == '') {
        $res = [
          'result' => false,
          'message' => 'User meta not found',
          'data' => null,
        ];
      } else {
        $res = [
          'result' => true,
          'message' => 'User meta found',
          'data' => $user_meta,
        ];
      }
      wp_send_json($res, 200);
    } else {
      wp_send_json([
        'success' => false,
        'message' => esc_html__('Please login to your account', ULTIMATE_AI_SLUG),
        'data' => null,
      ], 200);
    }
  }

  public function ultimate_ai_change_pass()
  {
    if (!isset($_POST['pass']) && empty($_POST['pass'])) {
      $response = [
        'message' => esc_html__('Please set a password', ULTIMATE_AI_SLUG),
        'success' => false,
      ];
      wp_send_json($response, 200);
    }
    if (!isset($_POST['key']) && empty($_POST['key'])) {
      $response = [
        'message' => esc_html__('Invalid activation key.', ULTIMATE_AI_SLUG),
        'success' => false,
      ];
      wp_send_json($response, 200);
    }

    $pass = Ultimate_AI_Filter_Controller($_POST['pass']);
    $key = Ultimate_AI_Filter_Controller($_POST['key']);

    $users = get_users([
      'meta_key' => 'user_activation_key',
      'meta_value' => $key,
    ]);

    if (empty($users)) {
      $response = [
        'message' => esc_html__('Invalid user.', ULTIMATE_AI_SLUG),
        'success' => false,
      ];
      wp_send_json($response, 200);
    } else {
      $user_id = $users[0]->ID;
      $login = $users[0]->user_login;
      update_user_meta($user_id, 'user_activation_key', 'activated');
      update_user_meta($user_id, 'user_active', true);
      wp_set_password($pass, $user_id);
      $signin = wp_signon([
        'user_login' => $login,
        'user_password' => $pass,
        'remember' => true,
      ]);

      if (is_wp_error($signin)) {
        $response = [
          'message' => esc_html__('Password setting failed.please try again later.', ULTIMATE_AI_SLUG),
          'success' => false,
        ];
      } else {
        $response = [
          'message' => esc_html__('Password successfully set.logging in...', ULTIMATE_AI_SLUG),
          'success' => true,
          'redirect_url' => home_url(),
        ];
      }
      wp_send_json($response, 200);
    }
  }

  public function ultimate_ai_get_current_user()
  {
    $user = wp_get_current_user();
    if (!$user->ID) {
      wp_send_json(['data' => 'No user found'], 400);
    }
    // remove password form response
    $user->data->user_pass = '';
    $user_manager = new Ultimate_AI_User_Manager();
    $user = $user_manager->get_full_user($user);
    wp_send_json(['data' => $user], 200);
  }

  public function ultimate_ai_add_subscriber()
  {
    // Check if admin
    $this->restrict_req_for_admin();

    $subscriber = Ultimate_AI_Filter_Controller($_POST['subscriber']);
    $add_user = $this->user_manager->add_subscriber($subscriber);

    wp_send_json($add_user, 200);
  }

  public function ultimate_ai_get_subscribers()
  {
    if (!current_user_can('administrator')) {
      wp_send_json([], 200);
    }
    $offset = isset($_POST['offset']) ? $_POST['offset'] : 0;
    $number = isset($_POST['limit']) ? $_POST['limit'] : 10;
    $search = isset($_POST['search']) ? $_POST['search'] : '';
    $args = [
      'search' => "*{$search}*",
      'number' => $number,
      'offset' => $offset,
      'meta_query' => [
        'relation' => 'OR',
        [
          'key' => 'ultimate_ai_user_meta',
          'value' => 's:10:"subscribed";b:1;',
          'compare' => 'LIKE',
        ],
        // array(
        //     'key' => 'user_activation_key',
        //     'value' => "NOT EXISTS",
        //     'compare' => 'NOT EXISTS'
        // )
      ],
    ];

    $user_res = [];
    $user_query = new WP_User_Query($args);
    $total = $user_query->get_total();
    if (!empty($user_query->get_results())) {
      foreach ($user_query->get_results() as $user) {
        $meta = get_user_meta($user->ID, 'ultimate_ai_user_meta', true);
        $user_res[] = [
          'user_name' => $user->display_name,
          'user_email' => $user->user_email,
          'avatar' => get_avatar($user->user_email, 36),
          'meta' => $meta,
        ];
      }
    }
    $res = [
      'subscribers' => $user_res,
      'total' => $total,
    ];

    wp_send_json($res, 200);
  }

  public function ultimate_ai_register_user()
  {
    $this->must_fields_error(['email', 'password', 'user_name']);
    $email = sanitize_email($_POST['email']);
    $user_name = sanitize_user($_POST['user_name']);
    $password = sanitize_text_field($_POST['password']);

    $user = $this->user_manager->register_user($user_name, $password, $email);

    if (is_wp_error($user) || !$user) {
      wp_send_json(['data' => esc_html__('User registration fail', ULTIMATE_AI_SLUG)], 400);
    }

    // gives free token if active
    $this->ultimate_ai_give_free_token_upon_registration($user);

    wp_send_json(['data' => ['status' => esc_html__('User registration complete.', ULTIMATE_AI_SLUG), 'user' => $user]], 201);
  }

  public function ultimate_ai_login_user()
  {
    $this->must_fields_error(['password', 'user_name']);
    $user_name = sanitize_user($_POST['user_name']);
    $password = sanitize_text_field($_POST['password']);

    $user = $this->user_manager->login_user($user_name, $password);

    if (is_wp_error($user) || !$user) {
      wp_send_json(['data' => esc_html__('User login fail', ULTIMATE_AI_SLUG)], 400);
    }

    wp_send_json(['data' => ['status' => esc_html__('User login complete.', ULTIMATE_AI_SLUG), 'user' => $user]], 200);
  }

  public function ultimate_ai_register_or_login_with_google()
  {
    $this->must_fields_error(['user_name', 'email']);
    $user_name = sanitize_user($_POST['user_name']);
    $email = sanitize_email($_POST['email']);
    $user = $this->user_manager->get_user_by('email', $email);

    if (!$user) {
      $is_3rd_party_login = true;
      $user = $this->user_manager->register_user($user_name, bin2hex(random_bytes(12)), $email, $is_3rd_party_login);
      $this->user_manager->update_user_meta($user, ['is_3rd_party_login' => true, 'login_method' => 'google']);

      // gives free token if active
      $this->ultimate_ai_give_free_token_upon_registration($user);
    } else {
      $this->user_manager->login_user_passowrd_less($user);
    }

    wp_send_json(['data' => ['status' => esc_html__('User login complete.', ULTIMATE_AI_SLUG), 'user' => $user]], 200);
  }

  public function ultimate_ai_user_exist_by()
  {
    $name = sanitize_text_field($_POST['name'] ?? '');
    $email = sanitize_email($_POST['email'] ?? '');
    $user_exist = $this->user_manager->get_user_by('login', $name);
    $res = ['name_exist' => $user_exist ? true : false];
    $user_exist = $this->user_manager->get_user_by('email', $email);
    $res['email_exist'] = $user_exist ? true : false;

    wp_send_json(['data' => $res], 200);
  }

  public function ultimate_ai_send_pass_reset()
  {
    if (isset($_POST['email']) && !empty($_POST['email'])) {
      $email = sanitize_email($_POST['email'] ?? '');
      $login_exist = $this->user_manager->get_user_by('login', $email);
      $email_exist = $this->user_manager->get_user_by('email', $email);
      if ($login_exist || $email_exist) {
        $user = $login_exist = $email_exist;
        $res = $this->user_manager->sendPasswordresetLink($user);
        wp_send_json($res, 200);
      } else {
        wp_send_json([
          'success' => false,
          'message' => esc_html__('User with this email/username not found', ULTIMATE_AI_SLUG),
        ], 200);
      }
    } else {
      wp_send_json([
        'success' => false,
        'message' => esc_html__('Please enter a username/email', ULTIMATE_AI_SLUG),
      ], 200);
    }
  }

  public function ultimate_ai_change_password()
  {
    $this->must_fields_error(['current_pass', 'new_pass']);
    $current_pass = Ultimate_AI_Filter_Controller($_POST['current_pass']);
    $new_pass = Ultimate_AI_Filter_Controller($_POST['new_pass']);
    $user = wp_get_current_user();
    // var_dump($user);
    if (!wp_check_password($current_pass, $user->user_pass)) {
      wp_send_json(['data' => __('Invalid current password', ULTIMATE_AI_SLUG)], 400);
    }

    $res = $this->user_manager->change_password($user, $new_pass);
    if (!$res) {
      return wp_send_json(['data' => __('Password change fail!', ULTIMATE_AI_SLUG)], 400);
    }

    return wp_send_json(['data' => $res], 200);
  }

  private function restrict_req_for_admin()
  {
    if (!current_user_can('administrator')) {
      wp_send_json('Not authorized', 401);
    }
  }

  private function must_fields_error(array $must_fields)
  {
    foreach ($must_fields as $field) {
      if (!$_POST[$field] || empty($_POST[$field])) {
        wp_send_json(['data' => esc_html__('Send proper information.', ULTIMATE_AI_SLUG)], 400);
      }
    }
  }
}

new Ultimate_AI_User_Controller();
